/*
* Copyright (C) Siemens AG 2013.
* All Rights Reserved. Confidential.
*
* Descr.:  Creates an instance of svg combo box
*          and handles all the functionality related to the combo box.
* <Description>
*
* Authors: Aalok Gokhale (aalok.gokhale@siemens.com)

/**
* Namespace for ComboBox.
* @namespace CPM
*/

var CPM = ( CPM || {} );

/**
* Creates an instance of ComboBox.
* @class
* @desc Class representing the svg combo box.
* @constructor
*/
CPM.ComboBox = function ( id, functionToCall, getNodes, comboBoxStyles, currentStyle ) {
    'use strict';

    var _id = id + '_ComboBox',
        _stylePropObj = CPM.StyleFactory.getCurrentStyleProps( currentStyle ),
        _isExtendedStyle = currentStyle.includes( CPM.Enums.StyleName.Extended ) ? true : false,
        _styles = {
            rx: _stylePropObj.ComboBox.Rx,
            'stroke-width': 0.5,
            stroke: _stylePropObj.ComboBox.Stroke
        },
        _strokeWidth,
        _comboButtonId = _id + '_ComboButton',
        _comboRectId = _id + '_ComboBoxRect',
        _optionsRectId = _id + '_ComboBoxOptionsRect',
        _clickableRectId = _id + '_OptionNumber_',
        _svgNodeId = _id + '_SvgOptionNode_',
        _clipPathId = _id + 'clippath',
        OPTIONS_RECT_PULLBACK = 1, // The options rect is pulled back up behind the comboBox rect so that no divide is seen between the comboBox and the options rect.
        _svgParent,
        _comboBoxDim = {},
    // Maximum height allowed for the dropdown.
        _maxOptionsHeight,
        _initialOptionsWidth,
        _ctrlHeight,
        _topCbOptions,
        _visibleRowLimit,
        _functionToCall = functionToCall,
        _getNodes = getNodes || _createTextNodes,
        _options = [],
        _selectedOption = '',
        _comboBoxGroup,
        _comboBoxButtonGroup,
        _selectedNodeGroup,
        _selctedNodeSVG,
        _selectedNode,
        _selectedNodeId = _clickableRectId + 'SelectedNode',
        _highlightedNode,
        _comboBoxOptionsGroup,
        _comboBoxContentGroup,
        _comboOptionsTranslateGroup,
        _clipPath,
        _clipRect,
        _cbClickableRectsGroup,
        _comboBoxRect,
        _comboBoxLine,
        _optionsRect,
        _optionsRectDim = {},
        _nodes = [],
        _svgOptionNodes = [],
        _cbScrollVerticalHandler,
        _isScrollbarCreated = false,
        _cbClickableRects = { nodes: [], selectedNode: undefined },
        _fontProps = {
            'font-family': 'Siemens Sans',
            'font-size': 14,
            'font-style': 'none',
            'font-weight': 400,
            'text-decoration': '',
            'stroke-width': 0,
            'fill': _stylePropObj.ComboBox.FontColor
        },
        _textHeight = CPM.svgUtil.getTextBBox( _fontProps, 'AA' ).height,
        self = this,
        _comboBoxOpen = false,

        _display = true,

        _scrollbarStyles = { stroke: _stylePropObj.ScrollBar.Stroke },
        _highlightScrollbarStyles = { stroke: _isExtendedStyle ? _stylePropObj.ScrollBar.ScrollerColor : 'none' },
        _triangleStyles = { fill: _stylePropObj.ComboBox.TriangleColor },
        _scrollRect = null,
        _scrollTriangle = null,

    //Scrollbar - Sb
        _optionsSb,
        _optionsSbId = _id + '_Scrollbar',
    // Scrollbar coordinates.
        _optionsSbTop,
        _optionsSbLeft,
        _optionsSbOffset,
        SCROLLBAR_WIDTH = 17,

    /**
    * Initializes styles for the combo box elements.
    * @function
    * @memberOf CPM.ComboBox
    * @param {styles} Styles to be applied to the combo box elements.
    */
        _initializeStyles = function ( styles ) {
            var attrib;
            if ( styles ) {
                if ( styles.comboBoxStyles ) {
                    for ( attrib in styles.comboBoxStyles ) {
                        _styles[attrib] = styles.comboBoxStyles[attrib];
                    }
                }
                _scrollbarStyles.rx = _styles.rx;
                _scrollbarStyles['stroke-width'] = _styles['stroke-width'];
                if ( styles.normalbuttonStyles ) {
                    for ( attrib in styles.normalbuttonStyles ) {
                        _scrollbarStyles[attrib] = styles.normalbuttonStyles[attrib];
                    }
                }
                if ( styles.pressedbuttonStyles ) {
                    for ( attrib in styles.pressedbuttonStyles ) {
                        _highlightScrollbarStyles[attrib] = styles.pressedbuttonStyles[attrib];
                    }
                }
            }
            _strokeWidth = _styles['stroke-width'];
        },

        _createTextNodes = function () {
            var i, length = _options.length, createSVGElement = CPM.svgUtil.createSVG, displayText, textNode, textNodes = { nodes: [] };
            _fontProps.x = 5;
            _fontProps.y = _comboBoxDim.height / 2 + _textHeight * 0.25;
            for ( i = 0; i < length; i++ ) {
                _fontProps.optiontext = _options[i];
                textNode = createSVGElement( 'text', _fontProps );
                if ( _comboBoxDim.width < _initialOptionsWidth ) {
                    displayText = CPM.svgUtil.truncateLabel( _options[i], _optionsRectDim.width, _fontProps );
                } else {
                    displayText = _options[i];
                }
                textNode.textContent = displayText;
                textNodes.nodes.push( textNode );
                // _fontProps.y += _textHeight*0.75;
            }
            textNodes.rectHeight = _comboBoxDim.height * length;
            //_fontProps.y = _optionsRectDim.height/2 + _textHeight * 0.25;
            textNodes.selectedNode = createSVGElement( 'text', _fontProps );
            textNodes.selectedNode.textContent = CPM.svgUtil.truncateLabel( _selectedOption, _comboBoxDim.width - 40, _fontProps ); //_selectedOption;
            return textNodes;
        },
        _createComboBoxBorder = function () {
            var createSVGElement = CPM.svgUtil.createSVG;
            _comboBoxLine = createSVGElement( 'line',
                {
                    x1: _comboBoxDim.left - 0.25,
                    x2: _comboBoxDim.left + _comboBoxDim.width + 0.25,
                    y1: _comboBoxDim.top + _comboBoxDim.height,
                    y2: _comboBoxDim.top + _comboBoxDim.height,
                    stroke: _stylePropObj.ComboBox.Line,
                    'stroke-width': '2px',
                    'id': 'ComboBoxLineId',
                    'appendTo': _comboBoxGroup
                } );
        },
    /**
    * Calculates the scrollbutton geometry.
    * @function
    * @memberOf CPM.ComboBox
    * @param {x} x position of the button.
    * @param {y} y position of the button
    * @param {width} Width of button.
    * @param {height} Height of the button.
    */
        _createScrollbutton = function ( x, y, width, height ) {
            var myTriPath, triangleEigth, attrib,
                cellHeight, offsY = 0, dataTifType;

            if ( height > width ) {
                offsY = ( height - width ) / 2;
                cellHeight = width;
            }
            else {
                cellHeight = height;
            }

            //this is used to construct the triangles for the buttons
            triangleEigth = cellHeight / 8;

            dataTifType = 'LineDown';
            myTriPath = 'M' + ( x + triangleEigth * 5 ) + ' ' + ( y + offsY + triangleEigth * 3 ) +
                    ' L' + ( x + cellHeight * 0.5 ) + ' ' + ( y + offsY + 5 * triangleEigth ) +
                    ' L' + ( x + triangleEigth * 3 ) + ' ' + ( y + offsY + triangleEigth * 3 ) + ' Z';

            // rect
            _scrollRect = CPM.svgUtil.createSVG( 'rect', {
                id: _comboButtonId,
                'data-tif-type': dataTifType,
                x: x,
                y: y,
                width: width,
                height: height,
                appendTo: _comboBoxButtonGroup
            } );
            for ( attrib in _scrollbarStyles ) {
                _scrollRect.setAttributeNS( null, attrib, _scrollbarStyles[attrib] );
            }
            // triangle
            _scrollTriangle = CPM.svgUtil.createSVG( 'path', {
                d: myTriPath,
                'pointer-events': 'none',
                appendTo: _comboBoxButtonGroup
            } );
            for ( attrib in _triangleStyles ) {
                _scrollTriangle.setAttributeNS( null, attrib, _triangleStyles[attrib] );
            }
        },

    /**
    * Updates the scrollbutton geometry.
    * @function
    * @memberOf CPM.ComboBox
    * @param {x} New x position of the button.
    * @param {y} New y position of the button
    * @param {width} New width of button.
    * @param {height} New height of the button.
    */
        _updateScrollButton = function ( x, y, width, height ) {
            var myTriPath, triangleEigth, cellHeight, offsY = 0,
            setAttributes = CPM.svgUtil.setAttr;

            if ( height > width ) {
                offsY = ( height - width ) / 2;
                cellHeight = width;
            }
            else {
                cellHeight = height;
            }

            //this is used to construct the triangles for the buttons
            triangleEigth = cellHeight / 8;

            myTriPath = 'M' + ( x + triangleEigth * 5 ) + ' ' + ( y + offsY + triangleEigth * 3 ) +
            ' L' + ( x + cellHeight * 0.5 ) + ' ' + ( y + offsY + 5 * triangleEigth ) +
            ' L' + ( x + triangleEigth * 3 ) + ' ' + ( y + offsY + triangleEigth * 3 ) + ' Z';

            // rect
            setAttributes( _scrollRect, {
                x: x,
                y: y,
                width: width,
                height: height
            } );

            // triangle
            setAttributes( _scrollTriangle, {
                d: myTriPath,
                'pointer-events': 'none'
            } );
        },

    /**
    * Function that handles the scrolling of the combo box scrollbar.
    * @function
    * @memberOf CPM.ComboBox
    * @param {changeType} Type of event - scrollStart, scrollChange, scrollEnd. (Redundant in scenario.)
    * @param {absValue} Absolute value of scroller position. (Redundant in scenario.)
    * @param {percValue} Percentage value of scroller position.
    */
        _onScrollEvent = function ( changeType, absValue, percValue ) {
            var originalRange = _optionsRectDim.height,
                currentRange = _maxOptionsHeight,
                rangeDifference = originalRange - currentRange;
            self.optionsMatrix.f = -1 * rangeDifference * percValue;
            _optionsSbOffset = percValue;
        },

    /**
    * Creates/updates the clipping by creating the clipRect. ClipRect cannot be simply updated with setAttributeNS as IE doesn't update the elements linked with the clipPath.
    * @function
    * @memberOf CPM.ComboBox
    */
        _createClipping = function () {
            if ( _clipPath.lastChild ) {
                _clipPath.removeChild( _clipPath.lastChild );
            } else {
                if ( !_clipRect ) {
                    _comboBoxContentGroup.setAttributeNS( null, 'clip-path', 'url(#' + _clipPathId + ')' );
                }
            }
            _clipRect = CPM.svgUtil.createSVG( 'rect',
                    {
                        'x': _optionsRectDim.left - _strokeWidth / 2,
                        'y': _optionsRectDim.top - _strokeWidth / 2,
                        'width': _optionsRectDim.width + _strokeWidth,
                        'height': _maxOptionsHeight + _strokeWidth,
                        'appendTo': _clipPath
                    } );
        },
    /**
    * Creates vertical scrollbar for combobox when height is less than control height.
    * @function
    * @memberOf CPM.ComboBox
    */
        _createVerticalScrollBar = function () {
            var
            left = _comboBoxDim.left + _comboBoxDim.width - _stylePropObj.ScrollBar.LeftPadding,
            top = _comboBoxDim.top + _comboBoxDim.height + _stylePropObj.ScrollBar.TopPadding,
            settings = {}, totalHeight, viewPortHeight;
            totalHeight = _options.length * _comboBoxDim.height;
            viewPortHeight = _ctrlHeight - _topCbOptions;
           ////Do not create vertical scrollbar if total height of tree is within viewport height or if viewport height is lesser than the node height i.e. height of one tree node.
            if ( viewPortHeight > totalHeight ) {
                if ( _isScrollbarCreated ) {
                    _cbScrollVerticalHandler.remove();
                    _isScrollbarCreated = false;
                }
                return;
            }
            settings.currentStyle = currentStyle;
            settings.width = _stylePropObj.ScrollBar.Width;
            settings.height = viewPortHeight + 10;//offset
            settings.height = settings.height < 0 ? 0 : settings.height;
            settings.height = settings.height - _stylePropObj.ScrollBar.HeightPadding;            
            settings.id = 'cb_';
            settings.startValue = 0;
            settings.endValue = 36;
            settings.initialScrollOffset = 0;
            settings.initialHeightPerc = 1 / _options.length;
            settings.scrollStep = 1 / _options.length;
            settings.horizontalSliderHeight = _stylePropObj.ScrollBar.HorizontalSliderHeight;
            _isScrollbarCreated = true;
            _cbScrollVerticalHandler.createScrollBar( settings, left, top, _svgParent, _onScrollEvent, false, currentStyle );
        },

    /**
    * Removes clipping.
    * @function
    * @memberOf CPM.ComboBox
    */
        _removeClipping = function () {
            _comboBoxContentGroup.removeAttributeNS( null, 'clip-path' );
            _clipRect.parentNode.removeChild( _clipRect );
            _clipRect = null;
        };

    /**
    * Vertical scrollbar for combobox
    * @function
    * @memberOf CPM.ComboBox
    */
    this.cbScrollVerticalHandler = null;

    /**
    * Sets the options of the combo box if nodes are not used. (TO BE EXTENDED)
    * @function
    * @memberOf CPM.ComboBox
    * @param {options} Array of texts/strings.
    */
    this.setOptions = function ( options ) {
        _options = options;
    };

    /**
    * Returns the maximum width the combobox should have based on the longest option.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.getMaxWidth = function () {
        var i, maxWidth = 0, width = 0;
        for ( i = 0; i < _options.length; i++ ) {
            width = CPM.svgUtil.getTextWidth( _fontProps, _options[i] );
            if ( width > maxWidth ) {
                maxWidth = width;
            }
        }
        return maxWidth + 45;//Including the combobox icon also.
    };

    /**
    * Sets the selected option of the combo box.
    * @function
    * @memberOf CPM.ComboBox
    * @param {option} Texts/string.
    */
    this.setSelectedOption = function ( option ) {
        var selectedNode, displayText, displayWidth;
        displayWidth = _comboBoxDim.width - 40;
        _selectedOption = option;
        _fontProps.y = _comboBoxDim.height / 2 + _textHeight * 0.25;
        _fontProps.x = 5;
        selectedNode = CPM.svgUtil.createSVG( 'text', _fontProps );
        if ( displayWidth < _initialOptionsWidth ) {
            displayText = CPM.svgUtil.truncateLabel( _selectedOption, displayWidth, _fontProps )
        }
        else {
            displayText = _selectedOption;
        }
        selectedNode.textContent = displayText;
        this.setSelectedNode( selectedNode );
    };

    /**
    * Clears the options of the combo box if nodes are not used. (TO BE EXTENDED)
    * @function
    * @memberOf CPM.ComboBox
    */
    this.clearOptions = function () {
        _options = [];
    };

    /**
    * Removes a certain option of the combo box. (TO BE EXTENDED)
    * @function
    * @memberOf CPM.ComboBox
    * @param {option} Texts/string to be removed.
    */
    this.removeOption = function ( option ) {
        var index = _options.indexOf( option );
        if ( index  > -1 ) {
            _options.splice( index, 1 );
        }
    };

    /**
    * Adds a certain option of the combo box. (TO BE EXTENDED)
    * @function
    * @memberOf CPM.ComboBox
    * @param {option} Texts/string to be removed.
    */
    this.addOption = function ( option, index ) {
        if ( _options.indexOf( option ) === -1 ) {
            if ( index === undefined || index > ( _options.length - 1 ) ) {
                _options.push( option );
            } else {
                _options.splice( index, 0, option );
            }
        }
    };

    /**
    * Set a visible row limit after which a scrollbar is to be used. (TO BE EXTENDED)
    * @function
    * @memberOf CPM.ComboBox
    * @param {visibleRowLimit} Number of rows to be visible.
    */
    this.setVisibleRowLimit = function ( visibleRowLimit ) {
        _visibleRowLimit = visibleRowLimit;
    };

    /**
    * Draws the combo box.
    * @function
    * @memberOf CPM.ComboBox
    * @param {parentNode} Parent node of the combo box.
    * @param {cbRect} Rect with coordinates of the combo box.
    * @param {optionsBottomLimit} Bottom limit till where space is available for the scrollbar to be drawn.
    */
    this.draw = function ( parentNode, cbRect, optionsBottomLimit ) {
        var createSVGElement = CPM.svgUtil.createSVG, attrib, defs;

        _svgParent = parentNode;
        _cbScrollVerticalHandler = new CPM.Control.ScrollHandler();
        this.cbScrollVerticalHandler = _cbScrollVerticalHandler;
        _comboBoxDim.left = cbRect.left;
        _comboBoxDim.top = cbRect.top;
        _comboBoxDim.height = cbRect.height;
        _comboBoxDim.width = cbRect.width;
        _initialOptionsWidth = cbRect.width;
        _ctrlHeight = cbRect.ctrlHeight;
        _topCbOptions = cbRect.topCbOptions;

        _optionsRectDim.left = cbRect.optionsLeft;
        _optionsRectDim.top = cbRect.top + cbRect.height - OPTIONS_RECT_PULLBACK;
        _optionsRectDim.width = cbRect.optionsWidth;
        _optionsRectDim.height = cbRect.height + OPTIONS_RECT_PULLBACK;

        _optionsSbTop = cbRect.top + cbRect.height;
        _optionsSbLeft = cbRect.optionsLeft + cbRect.optionsWidth - SCROLLBAR_WIDTH;

        _maxOptionsHeight = optionsBottomLimit - _optionsRectDim.top;

        _comboBoxGroup = createSVGElement( 'g', {
            'id': /*_id + */'_comboBoxElementsGroup',
            'display': _display ? 'initial' : 'none',
            'appendTo': _svgParent
        } );
        _comboBoxGroup.addEventListener( 'mouseout', self.onPointerMoveOut );

        defs = createSVGElement( 'defs' );

        _clipPath = createSVGElement( 'clipPath',
                            {
                                'id': _clipPathId,
                                'appendTo': defs
                            } );

        _svgParent.appendChild( defs );
        _comboBoxContentGroup = createSVGElement( 'g', {
            'id': _id + '_comboBoxContentGroup',
            display: 'none',
            stroke: 'none',
            'appendTo': _comboBoxGroup
        } );

        _optionsRect = createSVGElement( 'rect',
                                            {
                                                //'fill': backColor,
                                                //'fill-opacity': 1,
                                                x: _optionsRectDim.left,
                                                y: _optionsRectDim.top,
                                                height: _optionsRectDim.height,
                                                width: _optionsRectDim.width,
                                                fill: 'rgba(255,255,255,1)',
                                                stroke: 'black',
                                                'id': _optionsRectId,
                                                'appendTo': _comboBoxContentGroup
                                            } );
        for ( attrib in _styles ) {
            _optionsRect.setAttributeNS( null, attrib, _styles[attrib] );
            if( !_isExtendedStyle ){
                _optionsRect.setAttribute( 'stroke', _stylePropObj.ComboBox.OptionsStroke );
                _optionsRect.setAttribute( 'stroke-width', '1px' );
            }
        }

        // Translate group contains the elements that will move along with scrolling.
        _comboOptionsTranslateGroup = createSVGElement( 'g', {
            'id': _id + '_comboOptionsTranslateGroup',
            'transform': 'translate(0,0)',
            'appendTo': _comboBoxContentGroup
        } );
        this.optionsMatrix = _comboOptionsTranslateGroup.transform.baseVal.getItem( 0 ).matrix;
        _comboBoxRect = createSVGElement( 'rect',
                                            {
                                                //'fill': backColor,
                                                //'fill-opacity': 1,
                                                x: _comboBoxDim.left,
                                                y: _comboBoxDim.top,
                                                height: _comboBoxDim.height,
                                                width: _comboBoxDim.width,
                                                'stroke': 'black',
                                                'id': _comboRectId,
                                                'appendTo': _comboBoxGroup
                                            } );
        for ( attrib in _styles ) {
            _comboBoxRect.setAttributeNS( null, attrib, _styles[attrib] );
        }
        _selectedNodeGroup = createSVGElement( 'g', {
            'id': _id + '_SelectedNodeGroup',
            'appendTo': _comboBoxGroup
        } );
        _selctedNodeSVG = createSVGElement( 'svg', {
            'id': _id + '_SelectedNodeSVG',
            x: _comboBoxDim.left,
            y: _comboBoxDim.top,
            height: _comboBoxDim.height,
            width: _comboBoxDim.width,
            'appendTo': _selectedNodeGroup
        } );
        _cbClickableRects.selectedNode = createSVGElement( 'rect',
                                            {
                                                x: _comboBoxDim.left,
                                                y: _comboBoxDim.top,
                                                height: _comboBoxDim.height,
                                                width: _comboBoxDim.width,
                                                'stroke': 'none',
                                                'fill-opacity': 0,
                                                id: _selectedNodeId,
                                                'appendTo': _selectedNodeGroup
                                            } );
        if ( _selectedNode ) {
            while ( _selctedNodeSVG.firstChild ) {
                _selctedNodeSVG.removeChild( _selctedNodeSVG.firstChild );
            }
            _selctedNodeSVG.appendChild( _selectedNode );
        }

        _comboBoxButtonGroup = createSVGElement( 'g', {
            'id': _id + '_ComboBoxButtonGroup',
            'appendTo': _comboBoxGroup
        } );

        _createScrollbutton( cbRect.left + cbRect.width - cbRect.height, cbRect.top, cbRect.height, cbRect.height );
        if ( !_isExtendedStyle ) {
            _createComboBoxBorder();
        }
        _comboBoxOptionsGroup = createSVGElement( 'g', {
            'id': _id + '_comboBoxOptionsGroup',
            'appendTo': _comboOptionsTranslateGroup
        } );
        _cbClickableRectsGroup = createSVGElement( 'g', {
            'id': _id + '_CbClickableRectsGroup',
            'appendTo': _comboOptionsTranslateGroup
        } );
        cbRect = null;
    };

    /**
    * Updates the combo box.
    * @function
    * @memberOf CPM.ComboBox
    * @param {cbRect} Rect with coordinates of the combo box.
    * @param {optionsBottomLimit} Bottom limit till where space is available for the scrollbar to be drawn.
    * @param {updateSelectedOption} Indicates whether the current selected option of the combobox also needs to be updated.
    */
    this.update = function ( cbRect, optionsBottomLimit, updateSelectedOption ) {
        var setAttributes = CPM.svgUtil.setAttr, i, length, rectTop, rectHeight, maxOptionsHeight = _maxOptionsHeight;
        _stylePropObj = CPM.StyleFactory.getCurrentStyleProps( currentStyle );
        _isExtendedStyle = currentStyle.includes( CPM.Enums.StyleName.Extended ) ? true : false;
        if ( _comboBoxGroup.parentNode ) {
            _comboBoxGroup.parentNode.removeChild( _comboBoxGroup );
            _svgParent.appendChild( _comboBoxGroup );
        }
        _comboBoxDim.left = cbRect.left;
        _comboBoxDim.top = cbRect.top;
        _comboBoxDim.height = cbRect.height;
        _comboBoxDim.width = cbRect.width;
        //For some reason in IE, _data.Width is 0 when the control is initialized. Hence _initialOptionsWidth become a -ve value in that case in the above draw method.
        //The _initialOptionsWidth is updated again here when the resize call is received immediately after initialization. In resize call, the _data.Width value is correct.
        if ( _initialOptionsWidth < 0 ) {
            _initialOptionsWidth = cbRect.width;
        }
        _ctrlHeight = cbRect.ctrlHeight;
        _topCbOptions = cbRect.topCbOptions;

        _updateScrollButton( cbRect.left + cbRect.width - cbRect.height, cbRect.top, cbRect.height, cbRect.height );

        setAttributes( _comboBoxRect,
            {
                x: _comboBoxDim.left,
                y: _comboBoxDim.top,
                fill: _stylePropObj.ComboBox.RectColor,
                height: _comboBoxDim.height,
                width: _comboBoxDim.width
            } );
        if ( !_isExtendedStyle ) {
            setAttributes( _comboBoxLine,
            {
                x1: _comboBoxDim.left - 0.25,
                x2: _comboBoxDim.left + _comboBoxDim.width + 0.25,
                y1: _comboBoxDim.top + _comboBoxDim.height,
                y2: _comboBoxDim.top + _comboBoxDim.height,
                stroke: _stylePropObj.ComboBox.Line,
                'stroke-width': '2px'
            } );
        }
        setAttributes( _selctedNodeSVG,
            {
                x: _comboBoxDim.left,
                y: _comboBoxDim.top,
                height: _comboBoxDim.height,
                width: _comboBoxDim.width
            } );
        setAttributes( _cbClickableRects.selectedNode,
            {
                x: _comboBoxDim.left,
                y: _comboBoxDim.top,
                height: _comboBoxDim.height,
                width: _comboBoxDim.width
            } );

        _optionsRectDim.left = cbRect.optionsLeft;
        _optionsRectDim.top = cbRect.top + cbRect.height - OPTIONS_RECT_PULLBACK;
        _optionsRectDim.width = cbRect.optionsWidth;

        _optionsSbTop = cbRect.top + cbRect.height;
        _optionsSbLeft = cbRect.optionsLeft + cbRect.optionsWidth - SCROLLBAR_WIDTH;

        _maxOptionsHeight = optionsBottomLimit - _optionsRectDim.top;

        if ( _comboBoxOpen ) {
            setAttributes( _optionsRect,
            {
                x: _optionsRectDim.left,
                y: _optionsRectDim.top
            } );
            length = _svgOptionNodes.length;
            rectTop = _optionsSbTop;
            rectHeight = ( _optionsRectDim.height - OPTIONS_RECT_PULLBACK ) / length;
            for ( i = 0; i < length; i++ ) {
                setAttributes( _svgOptionNodes[i], {
                    x: _optionsRectDim.left,
                    y: rectTop
                } );
                setAttributes( _cbClickableRects.nodes[i], {
                    x: _optionsRectDim.left,
                    y: rectTop
                } );
                rectTop += rectHeight;
            }
            // If dropdown height is greater than allowed, clip contents and restrict height to maximum allowed; add scrollbar.
            if ( _maxOptionsHeight < _optionsRectDim.height ) {
                if ( maxOptionsHeight !== _maxOptionsHeight ) {
                    setAttributes( _optionsRect, {
                        height: _maxOptionsHeight
                    } );
                }
                // [NOTE]: Scrollbar is removed and created again since the update method provided does not allow for alteration of the scroller height.
                if ( _optionsSb ) {
                    _optionsSb.remove();
                    _optionsSb = null;
                }
                if ( _optionsSbOffset > 0 ) {
                    this.optionsMatrix.f = -1 * ( _optionsRectDim.height - _maxOptionsHeight ) * _optionsSbOffset;
                }
                _createVerticalScrollBar();
                //_optionsSb = new PWC.Lib.Canvas.ScrollBar( _optionsSbId, _comboBoxContentGroup, _optionsSbLeft, _optionsSbTop, SCROLLBAR_WIDTH, _maxOptionsHeight - OPTIONS_RECT_PULLBACK, 0, 500, ( _maxOptionsHeight / _optionsRectDim.height ), _optionsSbOffset, 0.1, 'top_bottom', _onScrollEvent, _scrollbarStyles.stroke );
                _createClipping();
            } else {
                if ( _optionsSb ) {
                    // If scrollbar exists when not required, remove scrollbar and set dropdown height as per calculations.
                    _optionsSb.remove();
                    _optionsSb = null;
                    this.optionsMatrix.f = 0;
                    _optionsSbOffset = 0;
                    setAttributes( _optionsRect, {
                        height: _optionsRectDim.height
                    } );
                    _removeClipping();
                }
            }
        } else {
            _optionsRectDim.height = cbRect.height + OPTIONS_RECT_PULLBACK;
            setAttributes( _optionsRect,
            {
                x: _optionsRectDim.left,
                y: _optionsRectDim.top,
                height: _optionsRectDim.height,
                width: _optionsRectDim.width
            } );
        }
        if ( updateSelectedOption ) {
            self.setSelectedOption( _selectedOption );
        }
        cbRect = null;
    };
    this.updateComboBoxStyle = function ( updatedStyle, styles ) {
        var attrib;
        currentStyle = updatedStyle;
        _stylePropObj = CPM.StyleFactory.getCurrentStyleProps( updatedStyle );
        _isExtendedStyle = updatedStyle.includes( CPM.Enums.StyleName.Extended ) ? true : false;

        _styles = {
            rx: _stylePropObj.ComboBox.Rx,
            'stroke-width': 0.5,
            stroke: _stylePropObj.ComboBox.Stroke
        };
        _scrollbarStyles = { stroke: _stylePropObj.ScrollBar.Stroke };
        _highlightScrollbarStyles = { stroke: _isExtendedStyle ? _stylePropObj.ScrollBar.ScrollerColor : 'none' };
        _triangleStyles = { fill: _stylePropObj.ComboBox.TriangleColor };

        _initializeStyles( styles );

        _fontProps.fill = _stylePropObj.ComboBox.FontColor;
        _selctedNodeSVG.firstChild.setAttribute( 'fill', _fontProps.fill );

        for ( attrib in _scrollbarStyles ) {
            if ( _scrollbarStyles.hasOwnProperty( attrib ) ) {
                _scrollRect.setAttributeNS( null, attrib, _scrollbarStyles[attrib] );
            }
        }
        for ( attrib in _triangleStyles ) {
            if ( _triangleStyles.hasOwnProperty( attrib ) ) {
                _scrollTriangle.setAttributeNS( null, attrib, _triangleStyles[attrib] );
            }
        }
        for ( attrib in _styles ) {
            if ( _styles.hasOwnProperty( attrib ) ) {
                _optionsRect.setAttributeNS( null, attrib, _styles[attrib] );
                if ( !_isExtendedStyle ) {
                    _optionsRect.setAttribute( 'x', _optionsRectDim.left );
                    _optionsRect.setAttribute( 'y', _optionsRectDim.top );
                    _optionsRect.setAttribute( 'height', _optionsRectDim.height );
                    _optionsRect.setAttribute( 'width', _optionsRectDim.width );
                    _optionsRect.setAttribute( 'stroke', _stylePropObj.ComboBox.OptionsStroke );
                    _optionsRect.setAttribute( 'stroke-width', '1px' );
                }
            }
        }
        for ( attrib in _styles ) {
            if ( _styles.hasOwnProperty( attrib ) ) {
                _comboBoxRect.setAttributeNS( null, attrib, _styles[attrib] );
            }
        }
        _comboBoxRect.setAttribute( 'fill', _stylePropObj.ComboBox.RectColor );
        if ( _comboBoxLine ) {
            _comboBoxLine.setAttribute( 'stroke', _stylePropObj.ComboBox.Line );
        } else {
            _createComboBoxBorder();
        }
    };
    /**
    * Checks if event occurred on a combo box element.
    * @function
    * @memberOf CPM.ComboBox
    * @param {targetId} Id of event's target.
    */
    this.idMatch = function ( targetId ) {
        return ( ( targetId === _comboButtonId ) || ( targetId === _comboRectId ) || ( targetId.indexOf( _clickableRectId ) > -1 ) );
    };

    /**
    * Checks if event occurred on the combo box scrollbar.
    * @function
    * @memberOf CPM.ComboBox
    * @param {targetId} Id of event's target.
    */
    this.sbIdMatch = function ( targetId ) {
        return ( targetId.indexOf( _optionsSbId ) > -1 );
    };

    /**
    * Returns the combo box scrollbar.
    * @function
    * @memberOf CPM.ComboBox
    * @returns {object} Scrollbar object.
    */
    this.getScrollBar = function () {
        return _optionsSb;
    };

    /**
    * Executes actions on the pointer down event.
    * @function
    * @memberOf CPM.ComboBox
    * @param {targetId} Id of event's target.
    * @returns {bool} State of the combo box options display.
    */
    this.onPointerDown = function ( targetId ) {
        var splitClickableId = targetId.split( _clickableRectId ), selectedIndex, attrib;
        if ( targetId === _comboButtonId || targetId === _selectedNodeId ) {
            for ( attrib in _highlightScrollbarStyles ) { // Only set the properties in _highlightScrollbarStyles which are available in _scrollbarStyles.
                if ( _scrollbarStyles.hasOwnProperty( attrib ) ) {
                    _scrollRect.removeAttributeNS( null, attrib );
                    _scrollRect.setAttributeNS( null, attrib, _highlightScrollbarStyles[attrib] );

                }
            }
        } else {
            if ( splitClickableId.length > 1 ) {
                selectedIndex = parseInt( splitClickableId[1], 0 );
                // self.setSelectedNode( _svgOptionNodes[selectedIndex].firstChild.cloneNode( true ) ); Cannot use this as the height of the combobox is not the same as height of the options. Fix the svg of selected node to the correct height ?
                _functionToCall( selectedIndex );
            }
        }
        this.toggleOptions();
        return _comboBoxOpen;
    };

    /**
    * Executes actions on the pointer up event.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.onPointerUp = function () {
        var attrib;
        for ( attrib in _highlightScrollbarStyles ) {
            if ( _scrollbarStyles.hasOwnProperty( attrib ) ) {
                _scrollRect.removeAttributeNS( null, attrib );
                _scrollRect.setAttributeNS( null, attrib, _scrollbarStyles[attrib] );

            }
        }
    };

    /**
    * Executes actions on the pointer move event.
    * @function
    * @memberOf CPM.ComboBox
    * @param {targetId} Id of event's target.
    */
    this.onPointerMove = function ( targetId ) {
        var splitClickableId = targetId.split( _clickableRectId ), selectedIndex;
        if ( ( splitClickableId.length > 1 ) && targetId !== _selectedNodeId ) {
            selectedIndex = parseInt( splitClickableId[1], 0 );
            if ( _highlightedNode ) {
                if ( _highlightedNode === _cbClickableRects.nodes[selectedIndex] ) {
                    return;
                } else {
                    _highlightedNode.setAttribute( 'fill-opacity', 0 );
                    _highlightedNode = null;
                }
            }
            _highlightedNode = _cbClickableRects.nodes[selectedIndex];
            _highlightedNode.setAttribute( 'fill-opacity', 0.5 );
        } else {
            if ( _highlightedNode ) {
                _highlightedNode.setAttribute( 'fill-opacity', 0 );
                _highlightedNode = null;
            }
        }
    };

    /**
    * Executes actions on the pointer move out event.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.onPointerMoveOut = function () {
        if ( _highlightedNode ) {
            _highlightedNode.setAttribute( 'fill-opacity', 0 );
            _highlightedNode = null;
        }
        CPM.Common.Tooltip.hide();
    };

    /**
    * Toggles the display/visibility of the combo box options.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.toggleOptions = function () {
        if ( _comboBoxOpen ) {
            if ( !_isExtendedStyle ) {
                _comboBoxLine.setAttribute( 'y1', _comboBoxDim.top + _comboBoxDim.height );
                _comboBoxLine.setAttribute( 'y2', _comboBoxDim.top + _comboBoxDim.height );
            }
            if ( _highlightedNode ) {
                _highlightedNode.setAttribute( 'fill-opacity', 0 );
                _highlightedNode = null;
            }
            this.closeComboBox();
        } else {
            _comboBoxOpen = true;
            if ( !_isExtendedStyle ) {
                _comboBoxLine.setAttribute( 'y1', _comboBoxDim.top );
                _comboBoxLine.setAttribute( 'y2', _comboBoxDim.top );
            }
            if ( _getNodes ) {
                self.setNodes( _getNodes() );
            }
            else {
                self.setNodes( _createTextNodes() );
            }
            _comboBoxContentGroup.setAttribute( 'display', 'initial' );
            this.optionsMatrix.f = 0;
            _optionsSbOffset = 0;
        }
    };

    /**
    * Closes the combo box options.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.closeComboBox = function () {
        _comboBoxOpen = false;
        if ( !_isExtendedStyle ) {
            _comboBoxLine.setAttribute( 'y1', _comboBoxDim.top + _comboBoxDim.height );
            _comboBoxLine.setAttribute( 'y2', _comboBoxDim.top + _comboBoxDim.height );
        }
        if ( _isScrollbarCreated ) {
            _cbScrollVerticalHandler.remove();
            _isScrollbarCreated = false;
        }
        if ( _optionsSb ) {
            _optionsSb.remove();
            _optionsSb = null;
        }
        if ( _clipRect ) {
            _removeClipping();
        }
        _comboBoxContentGroup.setAttribute( 'display', 'none' );
        this.optionsMatrix.f = 0;
        _optionsSbOffset = 0;
    };

    /**
    * Makes the combo box visible.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.show = function () {
        _display = true;
        if ( _comboBoxGroup ) {
            _comboBoxGroup.setAttributeNS( null, 'display', 'initial' );
            this.optionsMatrix.f = 0;
            _optionsSbOffset = 0;
        }
    };

    /**
    * Hides the combo box.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.hide = function () {
        _display = false;
        if ( _comboBoxOpen ) {
            this.closeComboBox();
        }
        if ( _comboBoxGroup ) {
            _comboBoxGroup.setAttributeNS( null, 'display', 'none' );
        }
    };

    /**
    * Sets the nodes of the combo box.
    * @function
    * @memberOf CPM.ComboBox
    * @param {nodeDetails} Object containing the nodes, height of the all the options to be displayed and the selected node.
    */
    this.setNodes = function ( nodeDetails ) {
        var i, length = _nodes.length, rectHeight, rectTop, createSVGElement = CPM.svgUtil.createSVG, setAttributes = CPM.svgUtil.setAttr, svgNode, drawScrollbar = false;
        for ( i = 0; i < length; i++ ) {
            if ( _nodes[i].parentNode ) {
                _nodes[i].parentNode.removeChild( _nodes[i] );
            }
        }
        if ( nodeDetails.rectHeight !== ( _optionsRectDim.height - OPTIONS_RECT_PULLBACK ) ) {
            _optionsRectDim.height = nodeDetails.rectHeight + OPTIONS_RECT_PULLBACK;
            if ( _maxOptionsHeight < _optionsRectDim.height ) {
                drawScrollbar = true;
                rectHeight = _maxOptionsHeight;
            } else {
                rectHeight = _optionsRectDim.height;
            }
            setAttributes( _optionsRect,
                {
                    height: rectHeight
                } );
        } else {
            if ( _maxOptionsHeight < _optionsRectDim.height ) {
                drawScrollbar = true;
            }
        }
        if ( drawScrollbar ) {
            if ( _optionsSb ) {
                _optionsSb.remove();
                _optionsSb = null;
            }
            _createVerticalScrollBar();
            // _optionsSb = new PWC.Lib.Canvas.ScrollBar( _optionsSbId, _comboBoxContentGroup, _optionsSbLeft, _optionsSbTop, SCROLLBAR_WIDTH, _maxOptionsHeight - OPTIONS_RECT_PULLBACK, 0, 500, ( _maxOptionsHeight / _optionsRectDim.height ), 0, 0.1, 'top_bottom', _onScrollEvent, _scrollbarStyles.stroke );
            _createClipping();
        }

        _nodes = nodeDetails.nodes;
        length = _nodes.length;
        rectHeight = ( _optionsRectDim.height - OPTIONS_RECT_PULLBACK ) / length;
        rectTop = _optionsRectDim.top + OPTIONS_RECT_PULLBACK;
        for ( i = 0; i < length; i++ ) {
            if ( _svgOptionNodes[i] ) {
                svgNode = _svgOptionNodes[i];
                setAttributes( svgNode, {
                    x: _optionsRectDim.left,
                    y: rectTop,
                    height: rectHeight,
                    width: _optionsRectDim.width,
                    id: _svgNodeId + i,
                    display: 'initial',
                    'appendTo': _comboBoxOptionsGroup
                } );
            } else {
                svgNode = createSVGElement( 'svg',
                                            {
                                                x: _optionsRectDim.left,
                                                y: rectTop,
                                                height: rectHeight,
                                                width: _optionsRectDim.width,
                                                id: _svgNodeId + i,
                                                'appendTo': _comboBoxOptionsGroup
                                            } );
                _svgOptionNodes.push( svgNode );
            }
            svgNode.appendChild( _nodes[i] );
            if ( _cbClickableRects.nodes[i] ) {
                setAttributes( _cbClickableRects.nodes[i], {
                    x: _optionsRectDim.left,
                    y: rectTop,
                    height: rectHeight,
                    width: _optionsRectDim.width,
                    id: _clickableRectId + i,
                    display: 'initial',
                    'appendTo': _cbClickableRectsGroup
                } );
            } else {
                _cbClickableRects.nodes.push( createSVGElement( 'rect',
                                            {
                                                x: _optionsRectDim.left,
                                                y: rectTop,
                                                height: rectHeight,
                                                width: _optionsRectDim.width,
                                                'stroke': 'none',
                                                'fill-opacity': 0,
                                                fill: _stylePropObj.ComboBox.ClickableRect,
                                                id: _clickableRectId + i,
                                                'appendTo': _cbClickableRectsGroup
                                            } ) );
            }
            rectTop += rectHeight;

        }
        while ( length < _comboBoxOptionsGroup.childNodes.length ) {
            _comboBoxOptionsGroup.removeChild( _comboBoxOptionsGroup.lastChild );
            length++;
        }
        length = _nodes.length;
        while ( length < _cbClickableRectsGroup.childNodes.length ) {
            _cbClickableRectsGroup.removeChild( _cbClickableRectsGroup.lastChild );
            length++;
        }
        if ( nodeDetails.selectedNode ) {
            this.setSelectedNode( nodeDetails.selectedNode );
        }
    };

    /**
    * Sets the selected node of the combo box.
    * @function
    * @memberOf CPM.ComboBox
    * @param {selectedNode} The group containing the selected node.
    */
    this.setSelectedNode = function ( selectedNode ) {
        _selectedNode = selectedNode;
        if ( _selctedNodeSVG ) {
            while ( _selctedNodeSVG.firstChild ) {
                _selctedNodeSVG.removeChild( _selctedNodeSVG.firstChild );
            }
            _selctedNodeSVG.appendChild( _selectedNode );
        }
    };

    /**
    * Cleans up the combo box.
    * @function
    * @memberOf CPM.ComboBox
    */
    this.cleanUp = function () {
        while ( _svgParent && _svgParent.lastChild ) {
            _svgParent.removeChild( _svgParent.lastChild );
        }
        _cbClickableRects.selectedNode =
        _cbClickableRects.nodes = null;
        this.optionsMatrix =
        _svgParent =
        _functionToCall =
        _getNodes =
        _options =
        _comboBoxGroup =
        _comboBoxButtonGroup =
        _selectedNodeGroup =
        _comboBoxOptionsGroup =
        _comboBoxContentGroup =
        _comboOptionsTranslateGroup =
        _cbClickableRectsGroup =
        _clipPath =
        _clipRect =
        _selctedNodeSVG =
        _selectedNode =
        _highlightedNode =
        _comboBoxRect =
        _comboBoxLine =
        _optionsRect =
        _optionsRectDim =
        _nodes =
        _svgOptionNodes =
        _cbClickableRects =
        _optionsSb =
        self = null;
    };

    _initializeStyles( comboBoxStyles );
};