﻿/* 
* Copyright (C) Siemens AG 2016.
* All Rights Reserved. Confidential.
*
* @namespace PWC.Core.Tooltip
* @desc Collection of functions to handle tooltip.
*
*/

var CPM = ( CPM || {} );
CPM.Common = ( CPM.Common || {} );

CPM.Common.Tooltip = ( function ( lib ) {
    'use strict';

    // PRIVATE MEMBERS
    var

    /** 
    * Indicates whether the tooltip is suspended (e.g. during interaction) or not.
    * @memberOf PWC.Core.Tooltip 
    */
    _tooltipSuspended = false
    ,

    /** 
    * Indicates whether the 'tooltip suspend' is suspended (e.g. during interaction) or not.
    * @memberOf PWC.Core.Tooltip 
    */
    _tooltipSuspendSuspended = false
    ,

    /** 
    * Holds a reference to the last tooltip timer.
    * @memberOf PWC.Core.Tooltip 
    */
    _lastToolTipTimer = null
    ,
    _desktopLayout,
    /**
    * Hide the tooltip element.
    * @memberOf PWC.Core.Tooltip 
    */
    _hide = function () {
        var ttItems = _getTooltipItems(),
            divTooltip = ttItems.domDivTooltip;

        // tooltip will hide after timeout
        if ( _tooltipSuspendSuspended ) {
            //lib.Debug.consolLog( lib.Debug.enumTraceMode.Info, 'Tooltip.hide() suspend suspended' );
            return;
        }

        //lib.Debug.consolLog( lib.Debug.enumTraceMode.Info, 'Tooltip.hide() hide' );       

        divTooltip.style.visibility = 'hidden';
    },
    _getTooltipItems = function () {
        return _desktopLayout.tooltipItems;
    };

    // end of private stuff

    // PUBLIC MEMBERS
    return {

        /**
        * Create the DOM items for the tooltip support.
        * @memberOf PWC.Core.Tooltip 
        * @param {Object} desktopLayout Reference to the desktopLayout in the data tree.
        */
        createItems: function ( desktopLayout ) {
            var ttItems;
            _desktopLayout = desktopLayout;
            _desktopLayout.tooltipItems = {};
            ttItems = _desktopLayout.tooltipItems;

            ttItems.domDivTooltip = lib.createDomElement( 'div', {
                id: 'ID_Tooltip',
                style: 'pointer-events:none; position:absolute; visibility:hidden; height:auto; width:auto; background-color: white; border: 1px solid black; border-radius: 4px; z-index:65535;',
                appendTo: _desktopLayout.domNode
            } );
            lib.createDomElement( 'p', {
                style: 'color:black; margin: 1px 2px;',
                textNode: '',
                appendTo: ttItems.domDivTooltip
            } );
            ttItems.domDivTooltip.style['userSelect'] = 'none';
            //lib.HmiTypes.Utils.setPrefixedStyle( ttItems.domDivTooltip, 'userSelect', 'none' );
        }
        ,

        /**
        * Show the tooltip of the given item.
        * @memberOf PWC.Core.Tooltip 
        * @param {Object} item Reference to the item (screen item) in the data tree.
        * @param {Number} x X position for the tooltip.
        * @param {Number} y Y position for the tooltip.
        * @param {String} stateText State text to show as tooltip (optional).
        */
        show: function ( x, y, stateText ) {
            var itemTooltipText,
                ttItems = _getTooltipItems(),
                divTooltip = ttItems.domDivTooltip,
                rect, diffHori, diffVert;


            if ( _tooltipSuspended && stateText === undefined ) {
                return;
            }

            // Special handling of state text which is shown as tooltip.
            //if ( stateText ) {
            //    // If there is a timeout running, stop it!
            //    _tooltipSuspendSuspended = true;
            //    if ( _lastToolTipTimer !== null ) {
            //        clearTimeout( _lastToolTipTimer );
            //    }

            //    _lastToolTipTimer = setTimeout( function () {
            //        //lib.Debug.consolLog( lib.Debug.enumTraceMode.Info, 'Tooltip.hide() called by timer' );
            //        _tooltipSuspendSuspended = false;
            //        _lastToolTipTimer = null;
            //        _hide();
            //    }, 1000 );
            //}

            //lib.Debug.consolLog(lib.Debug.enumTraceMode.Info, 'Tooltip.show() x:' + x.toString() + ' y:' + y.toString());

            itemTooltipText = ( stateText ) ? stateText : '';
            //if ( lib.Settings.showDeveloperTooltip ) {
            //    if ( itemTooltipText !== '' ) {
            //        itemTooltipText += '   ';
            //    }
            //    itemTooltipText += '>>>item: ';
            //    itemTooltipText += use_Prop.getPropRaw( item, 'Name' ) || '';
            //    itemTooltipText += ' ' + item.rtoid;
            //}

            //lib.Debug.consolLog( lib.Debug.enumTraceMode.Info, 'Tooltip.show() : ' + ( ( itemTooltipText !== '' ) ? itemTooltipText : 'Empty tooltip will not be shown!!!' ) );
            if ( itemTooltipText !== '' ) {
                divTooltip.firstChild.textContent = lib.converterBlanks2Nbsps( itemTooltipText );

                divTooltip.style.left = ( x + 7 ) + 'px';
                divTooltip.style.top = ( y + 11 ) + 'px';
                divTooltip.style.visibility = 'visible';

                // Shift the div to be fully displayed  
                if ( divTooltip.getBoundingClientRect ) {
                    rect = divTooltip.getBoundingClientRect();

                    diffHori = ( rect.right >= document.documentElement.clientWidth ) ? document.documentElement.clientWidth - rect.right : 7;
                    diffHori = ( rect.left + diffHori > 0 ) ? diffHori : -rect.left + 7;
                    diffHori = ( rect.left <= 0 ) ? -rect.left : diffHori;
                    diffVert = ( rect.bottom >= document.documentElement.clientHeight ) ? document.documentElement.clientHeight - rect.bottom : 11;
                    diffVert = ( rect.top + diffVert > 0 ) ? diffVert : -y + 2;
                    diffVert = ( rect.top <= 0 ) ? -rect.top : diffVert;

                    if ( ( diffHori !== 7 ) || ( diffVert !== 11 ) ) {
                        if ( ( diffHori !== 7 ) && ( diffVert !== 11 ) ) {
                            diffVert = ( diffVert >= -20 ) ? -25 : diffVert;
                        }

                        diffVert = ( rect.top + diffVert >= 5 ) ? diffVert : -rect.top + 7;

                        divTooltip.style.left = ( x + diffHori ) + 'px';
                        divTooltip.style.top = ( y + diffVert ) + 'px';
                    }
                }
            }
        }
        ,

        /**
        * Hide the tooltip element.
        * @memberOf PWC.Core.Tooltip 
        */
        hide: _hide
        ,

        /**
        * Suspend the tooltip functionality.
        * @memberOf PWC.Core.Tooltip 
        */
        suspend: function () {
            _tooltipSuspended = true;
            _hide();
        }
        ,

        /**
        * Resume the tooltip functionality.
        * @memberOf PWC.Core.Tooltip 
        */
        resume: function () {
            _tooltipSuspended = false;
        }

    };

}( CPM.svgUtil ) );
