﻿/* 
* Copyright (C) Siemens AG 2013.
* All Rights Reserved. Confidential.
*
* @namespace CPM.Lib.ScrollBar
* @desc ScrollBar implementation in SVG.
*
*/
var CPM = ( CPM || {} );
CPM.Lib = ( CPM.Lib || {} );


//id,parentNode,x,y,width,height,startValue,endValue,initialWidthPerc,initialOffset,scrollStep,scrollButtonLocations,functionToCall
CPM.Lib.ScrollBar = function (props, currentStyle, horizontalLine) {
    'use strict';
    // PRIVATE API
    var
        // self  = this,
        _id = props.id,
        parentNode = props.parentNode,
        x = props.x,
        y = props.y,
        width = props.width,
        height = props.height,
        initialHeightPerc = props.initialHeightPerc,
        initialOffset = props.initialOffset,
        _startValue = props.startValue,
        _endValue = props.endValue,
        _scrollButtonLocations = props.scrollButtonLocations,
        _stylePropObj = CPM.StyleFactory.getCurrentStyleProps(currentStyle),
        _isExtendedStyle = currentStyle === CPM.Enums.StyleName.Extended ? true : false,
       _scrollbarBackColor = _stylePropObj.ScrollBar.BgColor,
        _scrollerColor = _stylePropObj.ScrollBar.ScrollerColor,
        _highlightColor = _stylePropObj.ScrollBar.HighlightColor,
        _disabledFillColor = _stylePropObj.ScrollBar.DisabledFillColor,
        _disabledStrokeColor = _stylePropObj.ScrollBar.DisabledStrokeColor,
        _functionToCall = props.functionToCall,
        //additional properties to be used later
        _horizOrVertical = 'vertical', //specifies wether scrollbar is horizontal or vertical
        _cellHeight = width, //the height or width of the buttons on top or bottom of the scrollbar
        _scrollStatus = false, //indicates whether scrolling is active  
        _scrollStep = props.scrollStep, //this value indicates how much the slider will scroll when the arrow buttons are pressed, values are in percentage
        _scrollPage,    // This value indicates how much the slider will scroll when the scrollbar is pressed, values are in percentage
        _parentGroup = CPM.svgUtil.createSVG( 'g' ),

        _enabled = true, // Indicates whether the Scrollbar is enabled or not
        _values = { abs: '', perc: initialOffset },        // The offset of the scroller wrt to the scrollbar when scroller moves.
        _scrollbarEvtX, // Stores the x position of the mouse when scroller is moved by clicking the scrollbar
        _scrollbarEvtY, // Stores the y position of the mouse when scroller is moved by clicking the scrollbar

        _scrollbar = null,
        _scrollbarX = 0,
        _scrollbarY = 0,
        _scrollbarWidth = 0,
        _scrollbarHeight = 0,

        _scrollUpperButton = null,
        _scrollLowerButton = null,

        _scroller = null,
        _scrollerX = 0,
        _scrollerY = 0,
        _scrollerHeight = 0,
        _scrollerWidth = 0,
        _scrollerPadding = 3,
        _scrollerMin = 10,
        
        //color shade change
        _colorShadeChange = function ( rgba, lum ) {
            var rsRGB, gsRGB, bsRGB, asRGB = 1;
            rgba = rgba.split( 'rgba(' )[1].split( ',' );
            rsRGB = parseInt( rgba[0] );
            gsRGB = parseInt( rgba[1] );
            bsRGB = parseInt( rgba[2] );
            asRGB = rgba[3];
            rsRGB = Math.round( Math.min( Math.max( 0, rsRGB + ( rsRGB * lum ) ), 255 ) );
            gsRGB = Math.round( Math.min( Math.max( 0, gsRGB + ( gsRGB * lum ) ), 255 ) );
            bsRGB = Math.round( Math.min( Math.max( 0, bsRGB + ( bsRGB * lum ) ), 255 ) );

            return 'rgba(' + rsRGB + ',' + gsRGB + ',' + bsRGB + ',' + asRGB;
        },
        //change or add alpha value
        _changeAlpha = function ( rgba, alpha ) {
            if ( alpha >= 0 && alpha <= 1 ) {
                rgba = rgba.split( 'rgba(' )[1].split( ',' );
                return 'rgba(' + rgba[0] + ',' + rgba[1] + ',' + rgba[2] + ',' + alpha + ')';
            }
        },
        //create scrollbar geometry
        _createScrollbar = function (x, y, width, height, initialHeightPerc, initialOffset, currentStyle ) {
            var scrollUpperRectX, scrollLowerRectX,
                scrollUpperRectY, scrollLowerRectY, scrollBtnProps;
            //If scroller color available then use it.
            //if ( typeof scrollerRGBAColor === 'string' && scrollerRGBAColor.indexOf( 'rgba' ) !== -1 ) {
            //    CPM.Lib.scrollbarStyles.fill = _colorShadeChange( scrollerRGBAColor, 1.4 );
            //    CPM.Lib.highlightStyles.fill = _changeAlpha( scrollerRGBAColor, 0.8 );
            //    CPM.Lib.triangleStyles.fill = CPM.Lib.scrollerStyles.fill = _changeAlpha( scrollerRGBAColor, 0.5 );
            //}
            //first determine if vertical of horizontal
            if ( width > height ) {
                _horizOrVertical = 'horizontal';
                _cellHeight = height;
            }

            if ( _horizOrVertical === 'vertical' ) {
                _parentGroup.setAttribute( 'data-tif-type', 'VScroll' );
                _parentGroup.setAttribute( 'data-tif-id', 'VScroll_' + _id );
                scrollUpperRectX = x;
                scrollLowerRectX = x;
                switch ( _scrollButtonLocations ) {
                    case 'top_top':
                        _scrollbarX = x;
                        _scrollbarY = y + ( 2 * _cellHeight );
                        _scrollbarWidth = width;
                        _scrollbarHeight = height - ( 2 * _cellHeight );
                        scrollUpperRectY = y;
                        scrollLowerRectY = y + _cellHeight;
                        break;
                    case 'bottom_bottom':
                        _scrollbarX = x;
                        _scrollbarY = y;
                        _scrollbarWidth = width;
                        _scrollbarHeight = height - ( 2 * _cellHeight );
                        scrollUpperRectY = y + height - ( 2 * _cellHeight );
                        scrollLowerRectY = y + height - _cellHeight;
                        break;
                    case 'top_bottom':
                        _scrollbarX = x;
                        _scrollbarY = y + _cellHeight - 1;
                        _scrollbarWidth = width;
                        _scrollbarHeight = height - ( 2 * _cellHeight ) + 2;
                        scrollUpperRectY = y;
                        scrollLowerRectY = y + height - _cellHeight;
                        break;
                    default:
                        _scrollbarX = x;
                        _scrollbarY = y;
                        _scrollbarWidth = width;
                        _scrollbarHeight = height;
                        break;
                }
            }
            else {
                if ( _horizOrVertical === 'horizontal' ) {
                    _parentGroup.setAttribute( 'data-tif-type', 'HScroll' );
                    _parentGroup.setAttribute( 'data-tif-id', 'HScroll_' + _id );
                    scrollUpperRectY = y;
                    scrollLowerRectY = y;
                    switch ( _scrollButtonLocations ) {
                        case 'top_top':
                            _scrollbarX = x + ( 2 * _cellHeight );
                            _scrollbarY = y;
                            _scrollbarWidth = width - ( 2 * _cellHeight );
                            _scrollbarHeight = height;
                            scrollUpperRectX = x;
                            scrollLowerRectX = x + _cellHeight;
                            break;
                        case 'bottom_bottom':
                            _scrollbarX = x;
                            _scrollbarY = y;
                            _scrollbarWidth = width - ( 2 * _cellHeight );
                            _scrollbarHeight = height;
                            scrollUpperRectX = x + width - ( 2 * _cellHeight );
                            scrollLowerRectX = x + width - _cellHeight;
                            break;
                        case 'top_bottom':
                            _scrollbarX = x + _cellHeight - 1;
                            _scrollbarY = y;
                            _scrollbarWidth = width - ( 2 * _cellHeight ) + 2;
                            _scrollbarHeight = height;
                            scrollUpperRectX = x;
                            scrollLowerRectX = x + width - _cellHeight;
                            break;
                        default:
                            _scrollbarX = x;
                            _scrollbarY = y;
                            _scrollbarWidth = width;
                            _scrollbarHeight = height;
                            break;
                    }
                }
            }

            if (horizontalLine) {
                CPM.svgUtil.createSVG('line', {
                    x1: 0,
                    x2: props.width + CPM.Enums.Constants.alarmIconWidth + _stylePropObj.ScrollBar.Width, //scrollLowerRectX + (_cellHeight * 2) + 30,
                    y1: scrollLowerRectY,
                    y2: scrollLowerRectY,
                    'stroke-width': 1,
                    stroke: _stylePropObj.Control.HorizLineColor,
                    appendTo: _parentGroup
                });
            }
            _scrollbar = CPM.svgUtil.createSVG( 'rect', {
                id: 'scrollbar_' + _id,
                'data-tif-type': 'Scrollbar',
                x: _scrollbarX,
                y: _scrollbarY,
                'fill-opacity': _stylePropObj.ScrollBar.Opacity,
                width: ( _scrollbarWidth < 0 ) ? 0 : _scrollbarWidth,
                height: ( _scrollbarHeight < 0 ) ? 0 : _scrollbarHeight,
                appendTo: _parentGroup
            } );

            //_scrollbarBackColor
            _scrollbar.setAttributeNS(null, 'fill', _scrollbarBackColor);

            //now create scroller   
            if ( _horizOrVertical === 'vertical' ) {
                _scrollerY = _scrollbarY + ( _scrollbarHeight - _scrollbarHeight * initialHeightPerc ) * initialOffset;
                _scrollerHeight = Math.max( _scrollerMin, _scrollbarHeight * initialHeightPerc );
                _scrollPage = _scrollerHeight / ( _scrollbarHeight - _scrollerHeight );
                if ( ( _scrollerY + _scrollerHeight ) > ( _scrollbarY + _scrollbarHeight ) ) {
                    _scrollerY = _scrollbarY + _scrollbarHeight - _scrollerHeight;
                }

                _scroller = CPM.svgUtil.createSVG( 'rect', {
                    id: 'scroller_' + _id,
                    'data-tif-type': 'Thumber',
                    x: _isExtendedStyle ? _scrollbarX + _scrollerPadding : _scrollbarX,
                    y: _scrollerY,
                    width: _isExtendedStyle ? _scrollbarWidth - ( 2 * _scrollerPadding ) : _scrollbarWidth,
                    ry: _isExtendedStyle ? ( width / 4 ) : 0,
                    height: _scrollerHeight,
                    appendTo: _parentGroup
                } );
            }
            else {
                if ( _horizOrVertical === 'horizontal' ) {
                    _scrollerX = _scrollbarX + ( _scrollbarWidth - _scrollbarWidth * initialHeightPerc ) * initialOffset;
                    _scrollerWidth = Math.max( _scrollerMin, _scrollbarWidth * initialHeightPerc );
                    _scrollPage = _scrollerWidth / ( _scrollbarWidth - _scrollerWidth );
                    if ( ( _scrollerX + _scrollerWidth ) > ( _scrollbarX + _scrollbarWidth ) ) {
                        _scrollerX = _scrollbarX + _scrollbarWidth - _scrollerWidth;
                    }

                    _scroller = CPM.svgUtil.createSVG( 'rect', {
                        id: 'scroller_' + _id,
                        'data-tif-type': 'Thumber',
                        x: _scrollerX,
                        y: _isExtendedStyle ? _scrollbarY + _scrollerPadding : _scrollbarY,
                        width: _scrollerWidth,
                        height: _isExtendedStyle ? (_scrollbarHeight - (2 * _scrollerPadding) < 0) ? 0: _scrollbarHeight - (2 * _scrollerPadding): _scrollbarHeight,
                        ry: _isExtendedStyle ? (height / 4) : 0,
                        appendTo: _parentGroup
                    } );
                }
            }
            //_scrollerColor
            _scroller.setAttributeNS(null, 'fill', _scrollerColor);

            //append scrollbuttons
            if ( _scrollButtonLocations !== 'none_none' ) {
                scrollBtnProps = {
                    x: scrollUpperRectX,
                    y: scrollUpperRectY,
                    width: _cellHeight,
                    height: _cellHeight
                };
                //id,parentNode,scrollBtnProps,scrollButtonOrientation,currentStyle
                _scrollUpperButton = new CPM.Lib.ScrollButton( 'scrollUpperRect_' + _id, _parentGroup, scrollBtnProps, _horizOrVertical + '_up', currentStyle );
                scrollBtnProps = {
                    x: scrollLowerRectX,
                    y: scrollLowerRectY,
                    width: _cellHeight,
                    height: _cellHeight
                };
                //id,parentNode,scrollBtnProps,scrollButtonOrientation,currentStyle
                _scrollLowerButton = new CPM.Lib.ScrollButton('scrollLowerRect_' + _id, _parentGroup, scrollBtnProps, _horizOrVertical + '_down', currentStyle);
            }
        };

    // INITIALIZER
    ( function () {
        //create scrollbar
        if ( CPM.svgUtil.testParent( parentNode, _parentGroup ) ) {
            _createScrollbar(x, y, width, height, initialHeightPerc, initialOffset, currentStyle);
        }
    } )();

    // PUBLIC API
    return {
        onPointerDown: function ( x, y, target ) {
            if ( _enabled ) {
                var callerId = target.getAttributeNS( null, 'id' );
                if ( callerId === 'scroller_' + _id ) {
                    //case the mouse went down on scroller
                    this.scroll( x, y, 'mousedown' );
                }
                else if ( callerId === 'scrollLowerRect_' + _id || callerId === 'scrollUpperRect_' + _id ) {
                    //this part is for scrolling per button
                    this.buttonScrollActive = true;
                    this.scrollDir = 'down';
                    this.scrollStep = _scrollStep;
                    this.currentScrollButton = _scrollLowerButton;

                    if ( callerId === 'scrollUpperRect_' + _id ) {
                        this.scrollDir = 'up';
                        this.currentScrollButton = _scrollUpperButton;
                    }

                    //change appearance of button
                    this.currentScrollButton.onPointerDown();

                    this.scrollPercent( this.scrollDir, this.scrollStep );
                    //setTimeout( function () { self.scrollPerButton( self ); }, 400 );
                }
                else {
                    if ( callerId === 'scrollbar_' + _id ) {
                        //this part is for scrolling when mouse is down on scrollbar                
                        this.scrollbarScrollActive = true;
                        _scrollbarEvtX = x;
                        _scrollbarEvtY = y;
                        this.scrollDir = 'down';
                        this.scrollStep = _scrollStep;
                        if ( _horizOrVertical === 'vertical' ) {
                            if ( y < _scrollerY ) {
                                this.scrollDir = 'up';
                            }
                        }
                        else {
                            if ( _horizOrVertical === 'horizontal' && x < _scrollerX ) {
                                    this.scrollDir = 'up';
                            }
                        }
                        //change styling
                        //_scrollerColor
                        _scroller.removeAttributeNS(null, 'fill');
                        //_highlightColor
                        _scroller.setAttributeNS(null, 'fill', _highlightColor);
                        this.scrollPercent( this.scrollDir, _scrollPage );
                        //setTimeout( function () { self.scrollPerScrollbar( self ); }, 400 );
                    }
                }
            }
        },
        onPointerMove: function ( x, y ) {
            if ( _enabled ) {
                if ( _scrollStatus ) {
                    //this is for scrolling per scroller
                    this.scroll( x, y, 'mousemove' );
                    return true;
                }
            }
            return false;
        },
        onPointerUp: function ( x, y ) {
            if ( _enabled ) {
                //this is for finishing the different scroll modi
                if ( _scrollStatus ) {
                    //finishing scroll per scroller
                    this.scroll( x, y, 'mouseup' );
                }
                if ( this.buttonScrollActive ) {
                    //finishing scroll per button
                    this.buttonScrollActive = false;

                    //change appearance of button
                    this.currentScrollButton.onPointerUp();
                    this.currentScrollButton = undefined;
                }
                if ( this.scrollbarScrollActive ) {
                    //finishing scroll per scrollbar
                    this.scrollbarScrollActive = false;
                    //change styling
                    //_hightlightColor
                    _scroller.removeAttributeNS(null, 'fill');
                    //_scrollerColor
                    _scroller.setAttributeNS(null, 'fill', _scrollerColor);
                }
            }
        },

        scroll: function ( x, y, type ) {
            if ( type === 'mousedown' ) {
                _scrollStatus = true;
                this.panCoords = { x: x, y: y };
                this.fireFunction( 'scrollStart' );
                //change styling
                //_scrollerColor
                _scroller.removeAttributeNS(null, 'fill');
                //_hightlightColor
                _scroller.setAttributeNS(null, 'fill', _highlightColor);
            }
            else if ( type === 'mousemove' ) {
                var diffX = x - this.panCoords.x;
                var diffY = y - this.panCoords.y;
                var scrollerDiff = false;
                if ( _horizOrVertical === 'vertical' ) {
                    var scrollerOrigY = _scrollerY;
                    _scrollerY += diffY;
                    if ( _scrollerY < _scrollbarY ) {
                        _scrollerY = _scrollbarY;
                    }
                    if ( ( _scrollerY + _scrollerHeight ) > ( _scrollbarY + _scrollbarHeight ) ) {
                        _scrollerY = _scrollbarY + _scrollbarHeight - _scrollerHeight;
                    }
                    _scroller.setAttributeNS( null, 'y', _scrollerY );
                    if ( scrollerOrigY !== _scrollerY ) {
                        scrollerDiff = true;
                    }
                }
                if ( _horizOrVertical === 'horizontal' ) {
                    var scrollerOrigX = _scrollerX;
                    _scrollerX += diffX;
                    if ( _scrollerX < _scrollbarX ) {
                        _scrollerX = _scrollbarX;
                    }
                    if ( ( _scrollerX + _scrollerWidth ) > ( _scrollbarX + _scrollbarWidth ) ) {
                        _scrollerX = _scrollbarX + _scrollbarWidth - _scrollerWidth;
                    }
                    _scroller.setAttributeNS( null, 'x', _scrollerX );
                    if ( scrollerOrigX !== _scrollerX ) {
                        scrollerDiff = true;
                    }
                }
                if ( scrollerDiff ) {
                    this.fireFunction( 'scrollChange' );
                }
                this.panCoords = { x: x, y: y };
            }
            else {
                if ( type === 'mouseup' ) {
                    _scrollStatus = false;
                    //change styling
                    //_hightlightColor
                    _scroller.removeAttributeNS(null, 'fill');
                    //_scrollerColor
                    _scroller.setAttributeNS(null, 'fill', _scrollerColor);
                    this.fireFunction( 'scrollEnd' );
                }
            }
        },

        scrollPercent: function ( direction, increment ) {
            var currentValues = this.getValue();
            if ( direction === 'up' ) {
                increment = increment * -1;
            }
            var newPercent = currentValues.perc + increment;
            if ( newPercent < 0 ) {
                newPercent = 0;
            }
            if ( newPercent > 1 ) {
                newPercent = 1;
            }
            this.scrollToPercent( newPercent );
        },

        scrollToPercent: function ( percValue ) {
            var newX, newY;
            if ( percValue >= 0 && percValue <= 1 ) {
                if ( _horizOrVertical === 'vertical' ) {
                    newY = _scrollbarY + ( _scrollbarHeight - _scrollerHeight ) * percValue;
                    _scrollerY = newY;
                    _scroller.setAttributeNS( null, 'y', newY );
                }
                if ( _horizOrVertical === 'horizontal' ) {
                    newX = _scrollbarX + ( _scrollbarWidth - _scrollerWidth ) * percValue;
                    _scrollerX = newX;
                    _scroller.setAttributeNS( null, 'x', newX );
                }

                this.fireFunction( 'scrolledStep' );
            }
        },

        scrollPerButton: function ( self ) {
            if ( self.buttonScrollActive ) {
                self.scrollPercent( self.scrollDir, self.scrollStep );
                setTimeout( function () { self.scrollPerButton( self ); }, 50 );
            }
        },

        scrollPerScrollbar: function ( self ) {
            var scroll = true;
            if ( _horizOrVertical === 'vertical' ) {
                if ( ( self.scrollDir === 'up' && _scrollerY < _scrollbarEvtY ) || ( self.scrollDir === 'down' && _scrollerY + _scrollerHeight > _scrollbarEvtY ) ) {
                    scroll = false;
                }
            } else {
                if ( ( self.scrollDir === 'up' && _scrollerX < _scrollbarEvtX ) || ( self.scrollDir === 'down' && _scrollerX + _scrollerWidth > _scrollbarEvtX ) ) {
                    scroll = false;
                }
            }
            if ( self.scrollbarScrollActive && scroll ) {
                self.scrollPercent( self.scrollDir, _scrollPage );
                setTimeout( function () { self.scrollPerScrollbar( self ); }, 50 );
            }
        },

        scrollToValue: function ( value ) {
            if ( ( value >= _startValue && value <= _endValue ) || ( value <= _startValue && value >= _endValue ) ) {
                var percValue = value / ( _endValue - _startValue );
                this.scrollToPercent( percValue );
            }
        },

        getValue: function () {
            var perc;
            if ( _horizOrVertical === 'vertical' ) {
                perc = ( _scrollerY - _scrollbarY ) / ( _scrollbarHeight - _scrollerHeight );
            }
            if ( _horizOrVertical === 'horizontal' ) {
                perc = ( _scrollerX - _scrollbarX ) / ( _scrollbarWidth - _scrollerWidth );
            }
            var abs = _startValue + ( _endValue - _startValue ) * perc;
            return { 'abs': abs, 'perc': perc };
        },

        fireFunction: function ( changeType ) {
            _values = this.getValue();
            if ( typeof ( _functionToCall ) === 'function' ) {
                _functionToCall( changeType, _values.abs, _values.perc );
                return;
            }
            if ( typeof ( _functionToCall ) === 'object' ) {
                _functionToCall.scrollbarChanged( _id, changeType, _values.abs, _values.perc );
                return;
            }
        },

        updateScrollbar: function ( _x, _y, _width, _height, _initialHeightPerc ) {
            var scrollUpperRectX, scrollLowerRectX,
                scrollUpperRectY, scrollLowerRectY,
                setAttributes = CPM.svgUtil.setAttr;

            if ( _horizOrVertical === 'vertical' ) {

                scrollUpperRectX = _x;
                scrollLowerRectX = _x;
                switch ( _scrollButtonLocations ) {
                    case 'top_top':
                        _scrollbarX = _x;
                        _scrollbarY = _y + ( 2 * _cellHeight );
                        _scrollbarWidth = _width;
                        _scrollbarHeight = _height - ( 2 * _cellHeight );
                        scrollUpperRectY = _y;
                        scrollLowerRectY = _y + _cellHeight;
                        break;
                    case 'bottom_bottom':
                        _scrollbarX = _x;
                        _scrollbarY = _y;
                        _scrollbarWidth = _width;
                        _scrollbarHeight = _height - ( 2 * _cellHeight );
                        scrollUpperRectY = _y + _height - ( 2 * _cellHeight );
                        scrollLowerRectY = _y + _height - _cellHeight;
                        break;
                    case 'top_bottom':
                        _scrollbarX = _x;
                        _scrollbarY = _y + _cellHeight;
                        _scrollbarWidth = _width;
                        _scrollbarHeight = _height - ( 2 * _cellHeight );
                        scrollUpperRectY = _y;
                        scrollLowerRectY = _y + _height - _cellHeight;
                        break;
                    default:
                        _scrollbarX = _x;
                        _scrollbarY = _y;
                        _scrollbarWidth = _width;
                        _scrollbarHeight = _height;
                        break;
                }
            }
            else {
                if ( _horizOrVertical === 'horizontal' ) {

                    scrollUpperRectY = _y;
                    scrollLowerRectY = _y;
                    switch ( _scrollButtonLocations ) {
                        case 'top_top':
                            _scrollbarX = _x + ( 2 * _cellHeight );
                            _scrollbarY = _y;
                            _scrollbarWidth = _width - ( 2 * _cellHeight );
                            _scrollbarHeight = _height;
                            scrollUpperRectX = _x;
                            scrollLowerRectX = _x + _cellHeight;
                            break;
                        case 'bottom_bottom':
                            _scrollbarX = _x;
                            _scrollbarY = _y;
                            _scrollbarWidth = _width - ( 2 * _cellHeight );
                            _scrollbarHeight = _height;
                            scrollUpperRectX = _x + _width - ( 2 * _cellHeight );
                            scrollLowerRectX = _x + _width - _cellHeight;
                            break;
                        case 'top_bottom':
                            _scrollbarX = _x + _cellHeight;
                            _scrollbarY = _y;
                            _scrollbarWidth = _width - ( 2 * _cellHeight );
                            _scrollbarHeight = _height;
                            scrollUpperRectX = _x;
                            scrollLowerRectX = _x + _width - _cellHeight;
                            break;
                        default:
                            _scrollbarX = _x;
                            _scrollbarY = _y;
                            _scrollbarWidth = _width;
                            _scrollbarHeight = _height;
                            break;
                    }
                }
            }

            setAttributes( _scrollbar, {
                x: _scrollbarX,
                y: _scrollbarY,
                width: _scrollbarWidth,
                height: _scrollbarHeight
            } );

            //now update scroller   
            if ( _horizOrVertical === 'vertical' ) {
                _scrollerY = _scrollbarY + ( _scrollbarHeight - _scrollbarHeight * _initialHeightPerc ) * _values.perc;
                _scrollerHeight = Math.max( _scrollerMin, _scrollbarHeight * _initialHeightPerc );
                _scrollPage = _scrollerHeight / ( _scrollbarHeight - _scrollerHeight );
                if ( ( _scrollerY + _scrollerHeight ) > ( _scrollbarY + _scrollbarHeight ) ) {
                    _scrollerY = _scrollbarY + _scrollbarHeight - _scrollerHeight;
                }
                setAttributes( _scroller, {
                    x: _scrollbarX + _scrollerPadding,
                    y: _scrollerY,
                    width: _scrollbarWidth - ( 2 * _scrollerPadding ),
                    height: _scrollerHeight
                } );
            }
            else {
                if ( _horizOrVertical === 'horizontal' ) {
                    _scrollerX = _scrollbarX + ( _scrollbarWidth - _scrollbarWidth * _initialHeightPerc ) * _values.perc;
                    _scrollerWidth = Math.max( _scrollerMin, _scrollbarWidth * _initialHeightPerc );
                    _scrollPage = _scrollerWidth / ( _scrollbarWidth - _scrollerWidth );
                    if ( ( _scrollerX + _scrollerWidth ) > ( _scrollbarX + _scrollbarWidth ) ) {
                        _scrollerX = _scrollbarX + _scrollbarWidth - _scrollerWidth;
                    }

                    setAttributes( _scroller, {
                        x: _scrollerX,
                        y: _scrollbarY + _scrollerPadding,
                        width: _scrollerWidth,
                        height: _scrollbarHeight - ( 2 * _scrollerPadding )
                    } );
                }
            }

            //Update scrollbuttons
            if ( _scrollButtonLocations !== 'none_none' ) {
                //id,parentNode,x,y,width,height,scrollButtonOrientation,functionToCall
                _scrollUpperButton.updateScrollButton( scrollUpperRectX, scrollUpperRectY, _cellHeight, _cellHeight );

                //id,parentNode,x,y,width,height,scrollButtonOrientation,functionToCall
                _scrollLowerButton.updateScrollButton( scrollLowerRectX, scrollLowerRectY, _cellHeight, _cellHeight );
            }
        },

        hide: function () {
            _parentGroup.setAttributeNS( null, 'display', 'none' );
        },

        show: function () {
            _parentGroup.setAttributeNS( null, 'display', 'inherit' );
        },

        remove: function () {
            _parentGroup.parentNode.removeChild( _parentGroup );
        },

        scrollbarHeight: function () {
            return _scrollbarHeight;
        },

        scrollbarWidth: function () {
            return _scrollbarWidth;
        },

        enable: function () {
            _enabled = true;
            //change appearance of scroller and scrollbar
            //DisabledColor
            _scroller.removeAttributeNS(null, 'fill');
            _scroller.removeAttributeNS(null, 'stroke');
            //_ScrollerColor
            _scroller.setAttributeNS(null, 'fill', _scrollerColor);
            //_DisabledColor
            _scrollbar.removeAttributeNS(null, 'fill');
            _scrollbar.removeAttributeNS(null, 'stroke');
            //_scrollbarBackColor
            _scrollbar.setAttributeNS(null, 'fill', _scrollbarBackColor);
            _scrollUpperButton.enable();
            _scrollLowerButton.enable();
        },

        disable: function () {
            _enabled = false;
            //change appearance of scroller and scrollbar
            //_ScrollerColor
            _scroller.removeAttributeNS(null, 'fill');
            //_DisabledColor
            _scroller.setAttributeNS(null, 'fill', _disabledFillColor);
            _scroller.setAttributeNS(null, 'stroke', _disabledStrokeColor);
            //_scrollbarBackColor
            _scrollbar.removeAttributeNS(null, 'fill');
            //_disabledColor
            _scrollbar.setAttributeNS(null, 'fill', _disabledFillColor);
            _scrollbar.setAttributeNS(null, 'stroke', _disabledStrokeColor);
            _scrollUpperButton.disable();
            _scrollLowerButton.disable();
        },

        getState: function () {
            return _enabled;
        },

        updateScrollerColor: function ( scrollerRGBAColor ) {
            if ( typeof scrollerRGBAColor === 'string' && scrollerRGBAColor.indexOf( 'rgba' ) !== -1 ) {
                _scrollbarBackColor = _colorShadeChange(scrollerRGBAColor, 1.4);
                _highlightColor = _changeAlpha(scrollerRGBAColor, 0.8);
                if (_scrollbar) {
                    _scrollbar.setAttributeNS(null, 'fill', _scrollbarBackColor);
                }
                if (_scroller) {
                    _scroller.setAttributeNS(null, 'fill', _scrollerColor);
                }
                if ( _scrollUpperButton ) {
                    _scrollUpperButton.updateColor();
                }
                if ( _scrollLowerButton ) {
                    _scrollLowerButton.updateColor();
                }
            }
        }
    };
};